/*
    Generate a Radiance desciption of a superellipsoid

    Two parameters (eps1,eps2) control the curvature of the ellipse
    Special values of these parameters are
	-> 0     square
         < 1     cuboid
	   1     spherical, ellipsoid
           2     octahedron
	 > 2     pinched
    There is a smooth transition of forms between these values

    These objects are created from polygons, ribs and slices governs
    the number of polygons horizontally and vertically that are used
    to create the form. Typical values are 16 ribs and 16 slices.

    See the associated testscene for a range of these parameters

    The superellipsoid is centered at the origin and of unit size

    Mathematics
    The superellipsoid is defined as all the points lying on the surface
                         +-                               -+
                         | cos(theta)^eps1 * cos(phi)^eps2 |
         X(theta,phi) =  | cos(theta)^eps1 * sin(phi)^eps2 |
         ~               | sin(theta)^eps1                 |
                         +-                               -+

    Paul Bourke, Auckland University Department of Architecture
    November 1991
*/
#include <stdio.h>
#include <math.h>

#define PI       3.141592654
#define TWOPI    6.283185307
#define DTOR     0.01745329252
#define ABS(x)   ((x) < 0 ? -(x) : (x))
#define POW(x,y) ((x < 0 ? -1 : 1) * pow(ABS(x),y))

typedef struct {
    double x,y,z;
} XYZ;

double mypower(double,double);
void WriteFacet(XYZ *,int);

char *materialname;
char *objectname;
double eps1,eps2;
int ribs,slices;

main(argc,argv)
int argc;
char **argv;
{
    int f;
    double theta1,theta2,ctheta1,ctheta2,stheta1,stheta2;
    double phi1,phi2,cphi1,cphi2,sphi1,sphi2;
    XYZ p[4];

    if (argc < 7) {
	fprintf(stderr,"Usage: %s ",argv[0]);
	fprintf(stderr,"material name eps1 eps2 ribs slices ");
	fprintf(stderr,"\n");
	exit(1);
    }

    /* Extract the arguments */
    materialname  = argv[1];
    objectname    = argv[2];
    eps1   = ABS(atof(argv[3]));
    if (eps1 < 0.01)
	eps1 = 0.01;
    if (eps1 > 10)
	eps1 = 10;
    eps2   = ABS(atof(argv[4]));
    if (eps2 < 0.01)
        eps2 = 0.01;
    if (eps2 > 10)
	eps2 = 10;
    ribs   = ABS(atoi(argv[5]));
    if (ribs < 3) {
	fprintf(stderr,"Argument error from %s\n",argv[0]);
	fprintf(stderr,"Number of ribs must exceed 2\n");
	exit(1);
    }
    slices = ABS(atoi(argv[6]));
    if (slices < 3) {
        fprintf(stderr,"Argument error from %s\n",argv[0]);
        fprintf(stderr,"Number of slices must exceed 2\n");
        exit(1);
    }

    /* Echo the calling sequence */
    printf("# ");
    for (f=0;f<argc;f++)
	printf("%s ",argv[f]);
    printf("\n");

    /* Generate the superellipsoid */
    for (f=0;f<ribs*slices;f++) {
	theta1 = ( f      % ribs) * TWOPI / (double)ribs;
	theta2 = ((f + 1) % ribs) * TWOPI / (double)ribs;
	ctheta1 = POW(cos(theta1),eps1);
	ctheta2 = POW(cos(theta2),eps1);
	stheta1 = POW(sin(theta1),eps1);
	stheta2 = POW(sin(theta2),eps1);
	phi1 = ( f / ribs)        * PI / (double)slices;
	phi2 = ((f / ribs) + 1.0) * PI / (double)slices;
	cphi1 = 0.5 * POW(cos(phi1),eps2);
	cphi2 = 0.5 * POW(cos(phi2),eps2);
	sphi1 = 0.5 * POW(sin(phi1),eps2);
	sphi2 = 0.5 * POW(sin(phi2),eps2);
	
	if (f < ribs) {					    /* Top */
	    p[0].x = ctheta1 * sphi2;
	    p[1].x = ctheta2 * sphi2;
	    p[2].x = ctheta2 * sphi1;
	    p[0].y = stheta1 * sphi2;
	    p[1].y = stheta2 * sphi2;
	    p[2].y = stheta2 * sphi1;
	    p[0].z = cphi2;
	    p[1].z = cphi2;
	    p[2].z = cphi1;
	    WriteFacet(p,3);
	} else if (f < ribs*slices-ribs) {		    /* Middle */
	    p[0].x = ctheta1 * sphi2;
	    p[1].x = ctheta2 * sphi2;
	    p[2].x = ctheta2 * sphi1;
	    p[3].x = ctheta1 * sphi1;
	    p[0].y = stheta1 * sphi2;
	    p[1].y = stheta2 * sphi2;
	    p[2].y = stheta2 * sphi1;
	    p[3].y = stheta1 * sphi1;
	    p[0].z = cphi2;
	    p[1].z = cphi2;
	    p[2].z = cphi1;
	    p[3].z = cphi1;
	    WriteFacet(p,4);
	} else if (f < ribs*slices) {			    /* Bottom */
	    p[0].x = ctheta1 * sphi2;
	    p[1].x = ctheta2 * sphi1;
	    p[2].x = ctheta1 * sphi1;
	    p[0].y = stheta1 * sphi2;
	    p[1].y = stheta2 * sphi1;
	    p[2].y = stheta1 * sphi1;
	    p[0].z = cphi2;
	    p[1].z = cphi1;
	    p[2].z = cphi1;
	    WriteFacet(p,3);
	}
    }
    exit(0);
}

/*
    Write a facet in Radiance format, as a polygon
*/
void WriteFacet(p,n)
XYZ *p;
int n;
{
    int i;
    static int num;

    printf("%s polygon %s.%d\n",materialname,objectname,num++);
    printf("0\n0\n%ld\n",n*3);
    for (i=0;i<n;i++)
	printf("   %g %g %g\n",p[i].x,p[i].y,p[i].z);
    printf("\n");
}
